<?php

namespace App\Services;

use Razorpay\Api\Api;
use App\Models\PaymentGateway;

class RazorpayService
{
    protected $apiKey;
    protected $apiSecret;

    public function __construct()
    {
        $paymentMethod = PaymentGateway::where('type', 'razorpay')->first();

        if ($paymentMethod) {
            $config = json_decode($paymentMethod->config);
            $this->apiKey = $config->public_key ?? null;
            $this->apiSecret = $config->secret_key ?? null;
        }
    }

    public function paymentSuccess($order, $user, $amount)
    {
        $successUrl = route('payment.success');
        $cancelUrl  = route('payment.cancel');

        if (!$this->apiKey || !$this->apiSecret) {
            return response()->json([
                'message' => 'Razorpay credentials are not configured.'
            ], 400);
        }

        try {
            // Initialize Razorpay API
            $api = new Api($this->apiKey, $this->apiSecret);

            // Create Razorpay Order
            $razorpayOrder = $api->order->create([
                'receipt'         => 'order_' . $order->id,
                'amount'          => $amount * 100, // amount in paisa
                'currency'        => 'INR',
                'payment_capture' => 1 // auto capture
            ]);

            // Return data to frontend for Razorpay Checkout
            return [
                'razorpay_order_id' => $razorpayOrder['id'],
                'razorpay_key'      => $this->apiKey,
                'amount'            => $amount * 100,
                'currency'          => 'INR',
                'name'              => 'Order #' . $order->id,
                'description'       => 'Payment for Order #' . $order->id,
                'prefill'           => [
                    'name'  => $user->name,
                    'email' => $user->email,
                    'contact' => $user->phone ?? ''
                ],
                'callback_url' => $successUrl . '?payment=razorpay&order_id=' . $order->id,
                'cancel_url'   => $cancelUrl . '?payment=razorpay',
            ];

        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }
}
